/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.widgets;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.*;

public class Caret extends Widget {
	Canvas parent;
	int x, y, width, height;
	int blinkRate;
	boolean isVisible;
	Image image;
	Font font;

public Caret (Canvas parent, int style) {
	super (parent, style);
	this.parent = parent;
	createWidget ();
}

boolean blinkCaret () {
	if (!isVisible) return true;
	if (!object.visible) {
		object.visible = true;
		return true;
	}
	if (blinkRate == 0) return true;
	object.visible = false;
	return true;
}

void createHandle () {
	blinkRate = display.getCaretBlinkTime ();
	intrinsic.mx.containers.Canvas widget = new intrinsic.mx.containers.Canvas();
	widget.setStyle("backgroundColor", 0);
	widget.visible = false;
	object = widget;
}

void createWidget () {
	super.createWidget ();
	isVisible = true;
	if (parent.getCaret () == null) {
		parent.setCaret (this);
	}
}

public Rectangle getBounds () {
	checkWidget();
	if (image != null) {
		Rectangle rect = image.getBounds ();
		return new Rectangle (x, y, rect.width, rect.height);
	}
	return new Rectangle (x, y, width, height);
}

public Font getFont () {
	checkWidget();
	if (font == null) {
		return parent.getFont ();
	}
	return font;
}

public Image getImage () {
	checkWidget();
	return image;
}

public Point getLocation () {
	checkWidget();
	return new Point (x, y);
}

public Canvas getParent () {
	checkWidget();
	return parent;
}

public Point getSize () {
	checkWidget();
	if (image != null) {
		Rectangle rect = image.getBounds ();
		return new Point (rect.width, rect.height);
	}
	return new Point (width, height);
}

public boolean getVisible () {
	checkWidget ();
	return isVisible;
}

public boolean isVisible () {
	checkWidget();
	return object.visible && parent.isVisible () && parent.hasFocus ();
}

void killFocus () {
	if (display.currentCaret != this) return;
	display.setCurrentCaret (null);
	if (isVisible) object.visible = false;
}

void move (int x, int y) {
	object.x = x;
	object.y = y;
}

void resize (int width, int height) {
	if (image != null) {
		Rectangle rect = image.getBounds();
		width = rect.width;
		height = rect.height;
	}
	if (width == 0) width = 1;
	object.width = width;
	object.height = height;
}

void releaseHandle () {
	super.releaseHandle ();
	object = null;
	parent = null;
}

void releaseParent () {
	super.releaseParent ();
	if (this == parent.getCaret ()) parent.setCaret (null);
}

void releaseWidget () {
	super.releaseWidget ();
	image = null;
	font = null;
}

public void setBounds (int x, int y, int width, int height) {
	checkWidget();
	boolean samePosition = this.x == x && this.y == y;
	boolean sameExtent = this.width == width && this.height == height;
	if (samePosition && sameExtent) return;
	this.x = x;  this.y = y;
	this.width = width;  this.height = height;
	if (!sameExtent) resize (width, height);
	if (!samePosition) move (x, y);
}

public void setBounds (Rectangle rect) {
	if (rect == null) error (SWT.ERROR_NULL_ARGUMENT);
	setBounds (rect.x, rect.y, rect.width, rect.height);
}

public void setFont (Font font) {
	checkWidget();
	if (font != null && font.isDisposed ()) {
		error (SWT.ERROR_INVALID_ARGUMENT);
	}
	this.font = font;
}

void setFocus () {
	if (display.currentCaret == this) return;
	display.setCurrentCaret (this);
	if (isVisible) object.visible = true;
}

public void setImage (Image image) {
	checkWidget();
	if (image != null && image.isDisposed ()) {
		error (SWT.ERROR_INVALID_ARGUMENT);
	}
	this.image = image;
//	if (image != null) {
//		OS.Image_Source (imageHandle, image.handle);
//		OS.UIElement_Visibility (imageHandle, OS.Visibility_Visible);
//	} else {
//		OS.Image_Source (imageHandle, 0);
//		OS.UIElement_Visibility (imageHandle, OS.Visibility_Collapsed);
//	}
	resize (width, height);
}

public void setLocation (int x, int y) {
	checkWidget();
	if (this.x == x && this.y == y) return;
	this.x = x;  this.y = y;
	move (x, y);
}

public void setLocation (Point location) {
	checkWidget();
	if (location == null) error (SWT.ERROR_NULL_ARGUMENT);
	setLocation (location.x, location.y);
}

public void setSize (int width, int height) {
	checkWidget();
	if (this.width == width && this.height == height) return;
	this.width = width;  this.height = height;
	resize (width, height);
}

public void setSize (Point size) {
	checkWidget();
	if (size == null) error (SWT.ERROR_NULL_ARGUMENT);
	setSize (size.x, size.y);
}

public void setVisible (boolean visible) {
	checkWidget();
	if (visible == isVisible) return;
	isVisible = visible;
	if (!parent.hasFocus()) return;
	object.visible = visible;
}

}
